//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;
using System.Collections.Generic;
using System.Reflection;

using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Properties;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
	/// <summary>
	/// Represents a strategy that will transform a nested element into a collection of types.
	/// </summary>
    public class CollectionFactory<TObjectToCreate, TElementToCreate, TElementConfiguration, TObjectConfiguration> : ObjectFactoryBase<TObjectToCreate, TObjectConfiguration>
	{
		private readonly Type collectionType;
        private readonly IObjectFactory<TElementToCreate, TElementConfiguration> elementFactory;

		/// <summary>
		/// Initialize a new instance of the <see cref="CollectionTransformationStrategy"/> class with the type of collection to create and a transformation strategy for the nested element.
		/// </summary>
		/// <param name="collectionType">The type of collection to create.</param>		
		/// <param name="elementTransformationStrategy">The <see cref="TransformationStrategy"/> for the nested element.</param>
        public CollectionFactory(Type collectionType, IObjectFactory<TElementToCreate, TElementConfiguration> elementFactory)
		{
			if (collectionType == null) throw new ArgumentNullException("collectionType");
            if (elementFactory == null) throw new ArgumentNullException("elementFactory");
            if (!(typeof(ICollection<TElementToCreate>).IsAssignableFrom(collectionType)))
			{
				throw new ArgumentException(Resources.ExceptionTypeNotICollection, "collectionType");
			}
			
			this.collectionType = collectionType;
            this.elementFactory = elementFactory;
		}

		/// <summary>
		/// Gets the type of collection to create.
		/// </summary>
		/// <value>
		/// The type of collection to create.
		/// </value>
		public Type CollectionType
		{
			get { return collectionType; }
		}

		/// <summary>
		/// Gets the <see cref="TransformationStrategy"/> used for the nested element.
		/// </summary>
		/// <value>
		/// The <see cref="TransformationStrategy"/> used for the nested element.
		/// </value>
        public IObjectFactory<TElementToCreate, TElementConfiguration> ElementFactory
		{
            get { return elementFactory; }
        }

        public override TObjectToCreate Create(TObjectConfiguration objectConfiguration, IConfigurationSource configurationSource)
        {
            ICollection<TElementToCreate> result = null;

            if (objectConfiguration != null)
            {
                result = (ICollection<TElementToCreate>)Activator.CreateInstance(collectionType);

                IEnumerable<TElementConfiguration> configurations = (IEnumerable<TElementConfiguration>)objectConfiguration;

                foreach (TElementConfiguration configuration in configurations)
                {
                    result.Add(elementFactory.Create(configuration, configurationSource));
                }
            }

            return (TObjectToCreate)(object)result;
        }
    }
}
